import { InertialAnimation } from './IntertialAnimation';
import { DestructionHelper } from './DestructionHelper';
import {
    findParentComponent,
    getOrCreateScreenTransform,
    setRenderLayerRecursively,
    setRenderOrderRecursivelyRelativeToParent
} from './ComponentUtils';
import { LeaderboardConstants } from "./Interfaces/LeaderboardConstants";
import {
    BackgroundCustomization,
    EntryWrapper,
    IFullLeaderboard,
    ILeaderboardEntry,
    ILeaderboardEntryFallback,
    LeaderboardBackground,
    LeaderboardEntryBgTextureType, TimerTextureType
} from './Interfaces/LeaderboardRelated';
import { isSameUserEntry } from "./LeaderboardUtils";
import { BitmojiStickerLoader } from "./BitmojiStickerLoader";
import { FullLeaderboardAnimations } from "./FullLeaderboardAnimations";
import UserRecord = Leaderboard.UserRecord;
import { convertMilisecondsToDateString } from "./Modules/DateUtils";

export class FullLeaderboard implements IFullLeaderboard {
    protected scrollAnimation: InertialAnimation;
    protected readonly screenTransform;
    private readonly destructionHelper = new DestructionHelper();
    protected activeTouchId = null;
    private entriesParent: SceneObject = null;
    private leaderboardBackground: LeaderboardBackground;
    private userRecords: UserRecord[] = [];
    private currentUserRecord: UserRecord = null;
    private entryWrappers: EntryWrapper[] = [];
    private readonly PRELOAD_ENTRY_COUNT: number = LeaderboardConstants.FullLeaderboard.PRELOAD_ENTRY_COUNT;
    private appliedScroll: vec2 = vec2.zero();
    private isHeaderEnabled = false;
    private isFallbackInitialised: boolean = false;
    private fallbackEntriesParent: SceneObject = null;
    private isLeaderboardOpened: boolean = false;
    private outOfLeaderboardInteraction: InteractionComponent = null;
    private animations: FullLeaderboardAnimations = null;
    private isGlobalLeaderboard: boolean = false;

    constructor(private readonly scriptComponent: ScriptComponent,
                private readonly leaderboardBackgroundPrefab: ObjectPrefab,
                private readonly leaderboardEntryPrefab: ObjectPrefab,
                private readonly leaderboardEntryFallbackPrefab: ObjectPrefab,
                private readonly onLeaderboardClosedCallback: Function,
                private readonly onLeaderboardOpenedCallback: Function,
                private readonly renderOrder: number,
                private readonly bitmojiStickerLoader: BitmojiStickerLoader,
                backgroundCustomization: BackgroundCustomization) {

        this.screenTransform = getOrCreateScreenTransform(this.scriptComponent.getSceneObject(), this.destructionHelper);
        this.screenTransform.anchors.setCenter(LeaderboardConstants.FullLeaderboard.INITIAL_PARENT_POSITION);

        this.scrollAnimation = this.createInertialAnimation(scriptComponent,
            LeaderboardConstants.InertialAnimation.MAX_SPEED,
            LeaderboardConstants.InertialAnimation.FRICTION,
            LeaderboardConstants.InertialAnimation.SNAP_DURATION,
            LeaderboardConstants.InertialAnimation.SPRINGNESS);
        this.scrollAnimation.enabled = false;

        const leaderboardBackground = this.leaderboardBackgroundPrefab.instantiate(scriptComponent.getSceneObject());
        leaderboardBackground.enabled = true;

        this.leaderboardBackground = leaderboardBackground.getComponent("ScriptComponent") as LeaderboardBackground;
        this.leaderboardBackground.setHeaderMaskTopAnchors(LeaderboardConstants.FullLeaderboard.NO_HEADER_MASK_TOP_ANCHORS);
        this.leaderboardBackground.setHeaderAnchors(LeaderboardConstants.FullLeaderboard.PARENT_NO_HEADER_ANCHORS);
        this.leaderboardBackground.setBackgroundCustomizationOptionEnabled(backgroundCustomization);

        this.animations = new FullLeaderboardAnimations(this.screenTransform,
            this.leaderboardBackground.getTexturePreviewScreenTransform(),
            this.leaderboardBackground.getTopBorderScreenTransform(),
            this.leaderboardBackground.getRoundedCornersTransform(),
            this.leaderboardBackground.getTexturePreviewParentScreenTransform());

        const parentScreenTransform = this.leaderboardBackground.getEntriesParentScreenTransform();
        this.entriesParent = parentScreenTransform.getSceneObject();

        this.initClosingButton();
        this.initInteraction();

        this.leaderboardBackground.setPostEffectsEnabled(false);
    }

    public setLeaderboardName(name: string): void {
        this.leaderboardBackground.setLeaderboardName(name);
    }

    public setBitmoji(bitmojiTexture: Texture): void {
        this.leaderboardBackground.setFixedSizedTextureHeader(bitmojiTexture);
    }

    public setCustomTextureHeader(customTexture: Texture): void {
        const isScreenTexture = customTexture.control.isOfType("Provider.ScreenTextureProvider");
        this.animations.setIsScreenTextureInPreview(isScreenTexture);
        if (isScreenTexture) {
            this.leaderboardBackground.setResisableTextureHeader(customTexture);
        } else {
            this.leaderboardBackground.setFixedSizedTextureHeader(customTexture);
            this.leaderboardBackground.setBackgroundCustomizationOptionEnabled(BackgroundCustomization.Bitmoji);
        }
    }

    public show() {
        this.animateLeaderboard(true);
    }

    public hide() {
        this.animateLeaderboard(false);
    }

    public visualiseEntries(leaderboardRecordsWithCurrentUser: Leaderboard.UserRecord[], currentUserRecord: Leaderboard.UserRecord): void {
        this.userRecords = leaderboardRecordsWithCurrentUser;
        this.currentUserRecord = currentUserRecord;

        this.setFallbackEnabled(leaderboardRecordsWithCurrentUser.length <= 0);

        const entryWrappersNeededCount = Math.min(this.PRELOAD_ENTRY_COUNT, leaderboardRecordsWithCurrentUser.length);
        this.fillEntriesWrappersToSize(entryWrappersNeededCount);

        if (!isNull(currentUserRecord)) {
            this.enableHeaderIfPossible(currentUserRecord);
        }

        this.entryWrappers.forEach((entryWrapper, i) => {
            entryWrapper.representingIndex = i;
            this.updateEntryData(entryWrapper, this.userRecords[i]);
        });
    }

    public setFallbackEnabled(enabled: boolean) {
        if (enabled) {
            if (!this.isFallbackInitialised) {
                this.initialiseFallback();
            }
            this.leaderboardBackground.setHeaderEnabled(false);
            this.leaderboardBackground.setHeaderMaskTopAnchors(LeaderboardConstants.FullLeaderboard.NO_HEADER_MASK_TOP_ANCHORS);
            this.leaderboardBackground.setHeaderAnchors(LeaderboardConstants.FullLeaderboard.PARENT_NO_HEADER_ANCHORS);
        }

        if (!isNull(this.fallbackEntriesParent)) {
            this.fallbackEntriesParent.enabled = enabled;
        }
    }

    public setTimerUiEnabled(enabled: boolean): void {
        this.leaderboardBackground.setTimerUiEnabled(enabled);
    }

    public setTimerText(text: string): void {
        this.leaderboardBackground.setResetText(text);
    }

    public setTimeLeft(miliseconds: number): void {
        this.leaderboardBackground.setTimerTexture(miliseconds < 1000 * 60 * 60 ? TimerTextureType.Red : TimerTextureType.Regular);
        this.leaderboardBackground.setTimeLeftText(convertMilisecondsToDateString(miliseconds));
    }

    public setIsGlobalLeaderboard(isGlobalLeaderboard: boolean): void {
        this.isGlobalLeaderboard = isGlobalLeaderboard;
    }

    private getRealIndex(representingIndex: number, userRecord: Leaderboard.UserRecord): number {
        if (!this.isGlobalLeaderboard) {
            return representingIndex + 1;
        }

        if (!isNull(userRecord) && !isNull(userRecord.globalExactRank) && userRecord.globalExactRank !== 0) {
            return userRecord.globalExactRank;
        }
        return representingIndex + 1;
    }

    private initClosingButton(): void {
        const closingButtonInteraction = this.getOrCreateInteraction(this.leaderboardBackground.getClosingZoneScreenTransform().getSceneObject());
        closingButtonInteraction.isFilteredByDepth = true;

        let currentSameDirectionTouchCount = 0;
        let previousPos = null;

        closingButtonInteraction.onTouchStart.add((args) => {
            currentSameDirectionTouchCount = 0;
            previousPos = null;
        });

        closingButtonInteraction.onTouchMove.add((args) => {
            if (previousPos === null) {
                previousPos = args.position;
                return;
            }

            const isClosingDirection = previousPos.y < args.position.y;
            currentSameDirectionTouchCount = isClosingDirection ? currentSameDirectionTouchCount + 1 : 0;

            if (currentSameDirectionTouchCount >= LeaderboardConstants.FullLeaderboard.TOUCH_COUNT_TO_CLOSE_VIEW) {
                if (!this.animations.isPlaying()) {
                    this.animateLeaderboard(false);
                }
            }

            previousPos = args.position;
        });
    }

    private fillEntriesWrappersToSize(arraySize: number) {
        for (let i = this.entryWrappers.length; i < arraySize; i++) {
            this.entryWrappers.push(this.createEntry(i));
        }
        this.checkIfLeaderboardScrollable();
    }

    private createEntry(entryIndex: number): EntryWrapper {
        const currentEntriesCount = this.entriesParent.children.length;
        const topRelativeLimit: number = LeaderboardConstants.FullLeaderboard.ENTRIES_LIMIT_TOP_POSITION_Y;

        const bottomPosition: number = topRelativeLimit - (currentEntriesCount + 1) * LeaderboardConstants.FullLeaderboard.ENTRY_SIZE_Y;
        const topPosition: number = topRelativeLimit - currentEntriesCount * LeaderboardConstants.FullLeaderboard.ENTRY_SIZE_Y;

        const entry = this.leaderboardEntryPrefab.instantiate(this.entriesParent);
        entry.enabled = true;

        setRenderOrderRecursivelyRelativeToParent(entry, this.renderOrder);
        setRenderLayerRecursively(entry, this.entriesParent.layer);

        const screenTransform = entry.getComponent("ScreenTransform");
        screenTransform.anchors.top = topPosition;
        screenTransform.anchors.bottom = bottomPosition;

        const entryWrapper: EntryWrapper = {
            sceneObject: entry,
            leaderboardEntry: entry.getComponent("ScriptComponent") as ILeaderboardEntry,
            screenTransform: screenTransform,
            representingIndex: entryIndex,
            isCurrentUser: false,
            bitmojiRequestId: null
        };

        return entryWrapper;
    }

    private getOnComplete(toShow: boolean): () => void {
        if (!toShow) {
            return () => {
                this.onLeaderboardClosedCallback?.();
                this.outOfLeaderboardInteraction.enabled = false;
                this.isLeaderboardOpened = false;
            };
        } else {
            return () => {
                this.isLeaderboardOpened = true;
                this.outOfLeaderboardInteraction.enabled = true;
                this.onLeaderboardOpenedCallback?.();
            };
        }
    }
    private animateLeaderboard(toShow: boolean): void {
        this.animations.playAnimation(toShow, this.getOnComplete(toShow));
        this.leaderboardBackground.setPostEffectsEnabled(toShow);
    }

    private checkIfEntriesShouldBeShuffled(isScrollToTop: boolean): void {
        if (this.entryWrappers.length < 1) {
            return;
        }

        let lastEntry = this.entryWrappers[this.entryWrappers.length - 1];
        const indexAboveLimit = Math.floor(this.appliedScroll.y / LeaderboardConstants.FullLeaderboard.ENTRY_SIZE_Y);

        if (isScrollToTop) {
            while (indexAboveLimit > this.entryWrappers[0].representingIndex && lastEntry.representingIndex < this.userRecords.length - 1) {
                this.moveEntryFromHeadToTail(lastEntry);

                lastEntry = this.entryWrappers[this.entryWrappers.length - 1];
            }
        } else {
            let bottomEntryYCoord = lastEntry.screenTransform.anchors.getCenter().y;
            while (bottomEntryYCoord < LeaderboardConstants.FullLeaderboard.MIN_Y_COORD_TO_SHOW_ENTRY && this.entryWrappers[0].representingIndex > 0) {
                this.moveEntryFromTailToHead(this.entryWrappers[0]);
                bottomEntryYCoord = lastEntry.screenTransform.anchors.getCenter().y;
            }
        }
    }

    private moveEntryFromTailToHead(headEntry: EntryWrapper): void {
        const outOfBoundEntry = this.entryWrappers.pop();
        const positionToSet: vec2 = this.entryWrappers.length > 0 ?
            headEntry.screenTransform.anchors.getCenter().add(new vec2(0, LeaderboardConstants.FullLeaderboard.ENTRY_SIZE_Y))
            : outOfBoundEntry.screenTransform.anchors.getCenter();
        this.entryWrappers.unshift(outOfBoundEntry);

        outOfBoundEntry.representingIndex = headEntry.representingIndex - 1;
        outOfBoundEntry.screenTransform.anchors.setCenter(positionToSet);

        this.updateEntryData(outOfBoundEntry, this.userRecords[outOfBoundEntry.representingIndex]);
    }

    private moveEntryFromHeadToTail(tailEntry: EntryWrapper) {
        const outOfBoundEntry = this.entryWrappers.shift();
        const positionToSet: vec2 = this.entryWrappers.length > 0 ?
            tailEntry.screenTransform.anchors.getCenter().sub(new vec2(0, LeaderboardConstants.FullLeaderboard.ENTRY_SIZE_Y))
            : outOfBoundEntry.screenTransform.anchors.getCenter();

        this.entryWrappers.push(outOfBoundEntry);

        outOfBoundEntry.representingIndex = tailEntry.representingIndex + 1;
        outOfBoundEntry.screenTransform.anchors.setCenter(positionToSet);

        this.updateEntryData(outOfBoundEntry, this.userRecords[outOfBoundEntry.representingIndex]);
    }

    private updateEntryData(entry: EntryWrapper, userRecord: Leaderboard.UserRecord): void {
        entry.leaderboardEntry.setEntryIndex(this.getRealIndex(entry.representingIndex, userRecord));
        entry.leaderboardEntry.setUserRecord(userRecord, this.isGlobalLeaderboard);

        this.bitmojiStickerLoader.dismissToken(entry.bitmojiRequestId);

        if (!isNull(userRecord.snapchatUser)) {
            entry.bitmojiRequestId = this.bitmojiStickerLoader.generateToken();

            this.bitmojiStickerLoader.loadForUser(userRecord.snapchatUser, '').then((texture: Texture) => {
                if (this.bitmojiStickerLoader.isTokenValid(entry.bitmojiRequestId)) {
                    entry.leaderboardEntry.setBitmoji(texture);
                }
            })
                .catch(() => {
                    entry.leaderboardEntry.setBitmoji(null);
                });
        } else {
            entry.leaderboardEntry.setBitmoji(null);
        }

        const isCurrentUserEntry = isSameUserEntry(userRecord, this.currentUserRecord);
        if (entry.representingIndex === this.userRecords.length - 1) {
            if (isCurrentUserEntry) {
                entry.leaderboardEntry.setEntryBgTextureType(LeaderboardEntryBgTextureType.BottomSegmentBlue);
            } else {
                entry.leaderboardEntry.setEntryBgTextureType(LeaderboardEntryBgTextureType.BottomSegment);
            }
        } else {
            if (isCurrentUserEntry) {
                entry.leaderboardEntry.setEntryBgTextureType(LeaderboardEntryBgTextureType.MainSegmentBlue);
            } else {
                entry.leaderboardEntry.setEntryBgTextureType(LeaderboardEntryBgTextureType.MainSegment);
            }
        }
    }

    private checkIfLeaderboardScrollable(): void {
        this.scrollAnimation.enabled = this.entryWrappers.length >= LeaderboardConstants.FullLeaderboard.ENTRIES_NEEDED_FOR_SCROLL;
    }

    private enableHeaderIfPossible(currentUserRecord: UserRecord): void {
        if ((!isNull(currentUserRecord.globalRankPercentile) && currentUserRecord.globalRankPercentile !== 0)
            || (!isNull(currentUserRecord.globalExactRank) && currentUserRecord.globalExactRank !== 0)) {
            this.leaderboardBackground.setCurrentUserRecord(currentUserRecord);
            this.leaderboardBackground.setHeaderEnabled(true);
            this.leaderboardBackground.setHeaderAnchors(LeaderboardConstants.FullLeaderboard.PARENT_HEADER_ANCHORS);
            this.leaderboardBackground.setHeaderMaskTopAnchors(LeaderboardConstants.FullLeaderboard.HEADER_MASK_TOP_ANCHORS);
        } else {
            this.leaderboardBackground.setHeaderMaskTopAnchors(LeaderboardConstants.FullLeaderboard.NO_HEADER_MASK_TOP_ANCHORS);
            this.leaderboardBackground.setHeaderAnchors(LeaderboardConstants.FullLeaderboard.PARENT_NO_HEADER_ANCHORS);
        }
    }

    private createInertialAnimation(scriptComponent: ScriptComponent, maxSpeed: number, friction: number, snapDuration: number, springiness: number): InertialAnimation {
        return new InertialAnimation(scriptComponent, (delta) => {
            this.applyScroll(delta);
        }, () => {
        }, new vec2(0, 1), maxSpeed, friction, snapDuration);
    }

    private applyScroll(delta: vec2): void {
        if (this.entryWrappers.length <= 0) {
            return;
        }

        let offsetToApply = delta;
        if (delta.y > 0) {
            const bottomPositionInit = this.entryWrappers[this.entryWrappers.length - 1].screenTransform.anchors.getCenter();
            const bottomPositionAfterScroll = bottomPositionInit.add(delta);
            const bottomLimit = this.isHeaderEnabled ? LeaderboardConstants.FullLeaderboard.ENTRIES_BOTTOM_LIMIT_POSITION_HEADER : LeaderboardConstants.FullLeaderboard.ENTRIES_BOTTOM_LIMIT_POSITION_NO_HEADER;

            if (bottomPositionAfterScroll.y > bottomLimit) {
                offsetToApply = offsetToApply = new vec2(0, bottomLimit - bottomPositionInit.y);
            }
        } else {
            const topPositionInit = this.entryWrappers[0].screenTransform.anchors.getCenter();
            const topPositionAfterScroll = topPositionInit.add(delta);

            if (topPositionAfterScroll.y < LeaderboardConstants.FullLeaderboard.ENTRIES_TOP_LIMIT_POSITION) {
                offsetToApply = new vec2(0, LeaderboardConstants.FullLeaderboard.ENTRIES_TOP_LIMIT_POSITION - topPositionInit.y);
            }
        }

        this.applyOffset(offsetToApply);
        this.checkIfEntriesShouldBeShuffled(delta.y > 0);
    }

    private applyOffset(offset: vec2): void {
        this.entryWrappers.forEach((entryWrappers) => {
            let center = entryWrappers.screenTransform.anchors.getCenter();
            center = center.add(offset);
            entryWrappers.screenTransform.anchors.setCenter(center);

            if (center.y > LeaderboardConstants.FullLeaderboard.MAX_Y_COORD_TO_SHOW_ENTRY
                || center.y < LeaderboardConstants.FullLeaderboard.MIN_Y_COORD_TO_SHOW_ENTRY) {
                entryWrappers.sceneObject.enabled = false;
            } else {
                entryWrappers.sceneObject.enabled = true;
            }
        });

        this.appliedScroll = this.appliedScroll.add(offset);
    }

    private initInteraction(): void {
        const interaction = this.getOrCreateInteraction(this.leaderboardBackground.getTouchZoneSceneObject());
        interaction.isFilteredByDepth = true;

        interaction.onTouchStart.add(args => {
            if (!this.isInTouchZone(args.position)) return;
            if (this.activeTouchId == null) {
                this.activeTouchId = args.touchId;
                this.scrollAnimation.onTouchStart(this.screenPointtoLocalPoint(args.position));
            }
        });
        interaction.onTouchMove.add(args => {
            if (this.activeTouchId == args.touchId) {
                this.scrollAnimation.onTouchMove(this.screenPointtoLocalPoint(args.position));
            }
        });
        interaction.onTouchEnd.add(args => {
            if (this.activeTouchId == args.touchId) {
                this.activeTouchId = null;
                this.scrollAnimation.onTouchEnd(this.screenPointtoLocalPoint(args.position));
            }
        });

        this.outOfLeaderboardInteraction = this.getOrCreateInteraction(this.leaderboardBackground.getOutOfLeaderboardTouchZone().getSceneObject());
        this.outOfLeaderboardInteraction.isFilteredByDepth = true;
        this.outOfLeaderboardInteraction.onTap.add(() => {
            if (this.isLeaderboardOpened) {
                this.hide();
            }
        });

    }

    protected screenPointtoLocalPoint(screenPosition: vec2): vec2 {
        return this.screenTransform.screenPointToLocalPoint(screenPosition);
    }

    protected isInTouchZone(screenPos: vec2): boolean {
        return this.screenTransform.containsScreenPoint(screenPos);
    }

    protected getOrCreateInteraction(sceneObject: SceneObject): InteractionComponent {
        let interaction = sceneObject.getComponent('Component.InteractionComponent');
        if (isNull(interaction)) {
            interaction = this.destructionHelper.createComponent(sceneObject, 'Component.InteractionComponent');
            interaction.setCamera(findParentComponent(sceneObject, 'Component.Camera'));
            interaction.isFilteredByDepth = false;
            interaction.setMinimumTouchSize(0);
            const rmv = sceneObject.getComponent('Component.Image');
            if (!rmv) {
                // an invisible render mesh visual for proper touch handling by InteractionComponent
                const image = this.destructionHelper.createComponent(sceneObject, 'Component.Image');
                image.stretchMode = StretchMode.Stretch;
                interaction.addMeshVisual(image);
            }
        }
        return interaction;
    }

    private initialiseFallback() {
        this.isFallbackInitialised = true;
        this.fallbackEntriesParent = this.destructionHelper.createSceneObject(this.entriesParent.getParent(), "fallbackEntries");

        getOrCreateScreenTransform(this.fallbackEntriesParent, this.destructionHelper);

        for (let i = 0; i < 3; i++) {
            const currentEntriesCount = this.fallbackEntriesParent.children.length;
            const topRelativeLimit: number = LeaderboardConstants.FullLeaderboard.ENTRIES_LIMIT_TOP_POSITION_Y;

            const bottomPosition: number = topRelativeLimit - (currentEntriesCount + 1) * LeaderboardConstants.FullLeaderboard.ENTRY_SIZE_Y;
            const topPosition: number = topRelativeLimit - currentEntriesCount * LeaderboardConstants.FullLeaderboard.ENTRY_SIZE_Y;

            const entry = this.leaderboardEntryFallbackPrefab.instantiate(this.fallbackEntriesParent);
            entry.enabled = true;

            const leaderboardFallbackEntry = entry.getComponent("ScriptComponent") as ILeaderboardEntryFallback;
            leaderboardFallbackEntry.setEntryIndex(i);

            const screenTransform = entry.getComponent("ScreenTransform");
            screenTransform.anchors.top = topPosition;
            screenTransform.anchors.bottom = bottomPosition;
        }

        setRenderOrderRecursivelyRelativeToParent(this.fallbackEntriesParent, this.renderOrder);
        setRenderLayerRecursively(this.fallbackEntriesParent, this.entriesParent.layer);
    }

    reset(): void {
        this.userRecords = [];
        this.currentUserRecord = null;
        this.entryWrappers = [];

        this.entriesParent.children.forEach((entryScObj) => {
            entryScObj.destroy();
        });

        this.setFallbackEnabled(true);
    }
}
